<?php
/**
 * Plugin Name: wuk.ch KI Texte Cleaner
 * Plugin URI: https://wuk.ch/
 * Description: Entfernt AI-"Watermark"-/Fingerprint-Zeichen und Editor-Artefakte aus Inhalt & Builder-Daten.
 * Version: 1.1.0
 * Author: web updates kmu GmbH
 * Author URI: https://wuk.ch/
 *****************************
 * Update und Erweiterung um  34 “AI-Fingerprint”-Zeichen. Außerdem ist der Direktzugriff blockiert, JSON-Fehler 
 * beim Elementor-Hack werden abgefangen und die Ersetzungen sind klar gegliedert: löschen, zu Leerzeichen,
 * normalisieren (-, ", ').
 * https://thoka.network
 */

if (!defined('ABSPATH')) exit; // Direktzugriff blockieren

/*
 * Kern: Inhalt bereinigen
 */
add_filter('the_content', 'wuk_kitextfilter', 900);
add_filter('content_save_pre', 'wuk_kitextfilter');

function wuk_kitextfilter($content) {
	// 1) HTML-/Editor-Artefakte entfernen
	$content = preg_replace(
		array(
			'#\sdata\-(start|end)=(\\\"|\\\'|"|\')[0-9]*(\\\"|\\\'|"|\')#', // data-start / data-end
			"#\sdir=(\\\"|\\\'|'|\")auto(\\\"|\\\'|'|\")#",                // dir="auto"
		),
		'',
		$content
	);

	// 2) AI-"Fingerprints" & Invisible-Zeichen behandeln
	// 2a) Zeichen, die GELÖSCHT werden sollen (unsichtbare Steuerzeichen etc.)
	$delete = array(
		"\xE2\x80\x8B", // U+200B ZWSP
		"\xE2\x81\xA0", // U+2060 WJ (Word Joiner)
		"\xE2\x80\x8C", // U+200C ZWNJ
		"\xE2\x80\x8D", // U+200D ZWJ
		"\xE2\x80\x8E", // U+200E LRM
		"\xE2\x80\x8F", // U+200F RLM
		"\xEF\xBB\xBF", // U+FEFF BOM / ZW NBSP
		"\xE2\x81\xA1", // U+2061 FUNC APP
		"\xE2\x81\xA2", // U+2062 INVIS ×
		"\xE2\x81\xA3", // U+2063 INVIS SEP
		"\xE2\x81\xA4", // U+2064 INVIS +
		"\xE1\xA0\x9E", // U+181E MVS (Mongolian Vowel Separator)
		"\xE3\x85\xA4", // U+3164 Hangul Filler
		"\xC2\xAD",     // U+00AD Soft Hyphen
		"\xE2\x80\xAE", // U+202E RLO (RTL Override)
		"\xE2\xA0\x80", // U+2800 Braille Blank
		"\xE2\x80\x8A", // U+200A Hair Space (laut Liste: löschen)
	);

	// 2b) Zeichen, die zu LEERZEICHEN normalisiert werden
	$toSpace = array(
		"\xE2\x80\x80", // U+2000 Punctuation Space
		"\xE2\x80\x81", // U+2001 Em Quad / Em Space
		"\xE2\x80\x82", // U+2002 En Space
		"\xE2\x80\x83", // U+2003 Em Space
		"\xE2\x80\x84", // U+2004 Three-Per-Em Space
		"\xE2\x80\x85", // U+2005 Four-Per-Em Space
		"\xE2\x80\x86", // U+2006 Six-Per-Em Space
		"\xE2\x80\x87", // U+2007 Figure Space
		"\xE2\x80\x88", // U+2008 Punctuation Space
		"\xE2\x80\x89", // U+2009 Thin Space
		"\xC2\xA0",     // U+00A0 NBSP
		"\xE2\x80\xAF", // U+202F Narrow NBSP
	);

	// 2c) Zeichen, die zu Standard-Typografie normalisiert werden
	$normalize = array(
		// Striche / Bindestriche
		"\xE2\x80\x93" => '-', // U+2013 EN Dash
		"\xE2\x80\x94" => '-', // U+2014 EM Dash
		"\xE2\xB8\xBB" => '---', // U+2E3B THREE-EM DASH -> '---'
		"\xE2\x88\x92" => '-', // U+2212 Minus Sign
		"\xE2\x80\x91" => '-', // U+2011 Non-breaking hyphen
		// Anführungszeichen
		"\xE2\x80\x9C" => '"', // U+201C Left Double Quote
		"\xE2\x80\x9D" => '"', // U+201D Right Double Quote
		"\xE2\x80\x98" => "'", // U+2018 Left Single Quote
		"\xE2\x80\x99" => "'", // U+2019 Right Single Quote
		"\xCA\xBC"     => "'", // U+02BC Modifier Apostrophe
	);

	// löschen
	if (!empty($delete)) {
		$content = str_replace($delete, '', $content);
	}

	// zu Leerzeichen
	if (!empty($toSpace)) {
		$content = str_replace($toSpace, ' ', $content);
		// Mehrfach-Whitespace auf eins reduzieren (aber Zeilenumbrüche erhalten)
		$content = preg_replace('/[^\S\r\n]{2,}/', ' ', $content);
	}

	// normalisieren
	if (!empty($normalize)) {
		$content = strtr($content, $normalize);
	}

	return $content;
}

/*
 * Builder-spezifische Korrekturen (nur im Admin-Speicherprozess)
 */
add_action('save_post', function ($post_id, $post, $update) {
	if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
	if (!is_admin()) return;

	// Enfold / Avia
	$meta = get_post_meta($post_id, '_aviaLayoutBuilderCleanData', true);
	if ($meta) {
		$clean = wuk_kitextfilter($meta);
		update_post_meta($post_id, '_aviaLayoutBuilderCleanData', $clean);
	}

	// Elementor (JSON rekursiv bereinigen)
	$raw = get_post_meta($post_id, '_elementor_data', true);
	if ($raw) {
		$data = json_decode($raw, true);
		if (json_last_error() === JSON_ERROR_NONE && is_array($data)) {
			array_walk_recursive($data, function (&$value) {
				if (is_string($value) && $value !== '') {
					$value = wuk_kitextfilter($value);
				}
			});
			$clean = wp_slash(json_encode($data, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE));
			update_post_meta($post_id, '_elementor_data', $clean);
		}
	}

	// WPBakery benötigt keinen Hack
}, 50, 3);
